<?php

namespace App\Http\Controllers\Admin\Academy;

use App\Http\Controllers\Controller;
use App\Libraries\Rules\mobileValidation;
use App\Libraries\Rules\nationalCode;
use App\Models\Academy\academy;
use App\Models\Academy\agent;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class AcademyController extends Controller
{
    public function index()
    {
        $this->authorize('academy_index');

        if (request()->has('page')) {
            $list = academy::search()->withCount('student')->latest()->paginate();

            return response()->json($list);
        }

        return view('admin.Academy.academy.index');
    }

    public function create()
    {
        $this->authorize('academy_store');

        return view('admin.Academy.academy.create');
    }

    public function store(Request $request)
    {
        $this->authorize('academy_store');

        $request->offsetSet('birthdayOwner',toGorgian($request->input('birthdayOwner')));
        $request->offsetSet('birthdayAdmin',toGorgian($request->input('birthdayAdmin')));

        $this->validate($request, [
            'user_id' => 'required|numeric|exists:users,id',
            'area_id' => 'required|numeric|exists:areas,id',

            'firstNameOwner' => 'nullable|string|min:2|max:256',
            'lastNameOwner' => 'nullable|string|min:2|max:256',
            'nationalCodeOwner' => ['nullable', 'numeric', new nationalCode()],
            'birthdayOwner' => 'nullable|date',
            'birthplaceOwner' => 'nullable|string|min:5|max:256',
            'mobileOwner' => ['nullable', 'numeric', new mobileValidation()],
            'nationalCardOwner' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',
            'firstNameAdmin' => 'nullable|string|min:5|max:256',
            'lastNameAdmin' => 'nullable|string|min:5|max:256',
            'nationalCodeAdmin' => ['nullable', 'numeric', new nationalCode()],
            'birthdayAdmin' => 'nullable|date',
            'birthplaceAdmin' => 'nullable|string|min:5|max:256',
            'mobileAdmin' => ['nullable', 'numeric', new mobileValidation()],
            'nationalCardAdmin' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',

            'title' => 'required|string|min:1|max:1024',
            'email' => 'nullable|string|email|min:3|max:512',
            'fax' => 'nullable|digits_between:4,32',
            'website' => 'nullable|string|min:1|max:512',
            'telephone' => 'required|digits_between:10,16',
            'address' => 'required|string|min:1|max:2048',
            'postalCode' => 'required|digits_between:9,16',
            'telegram' => 'nullable|string|min:1|max:256',
            'whatsapp' => 'nullable|digits_between:10,16',
            'referalCode' => 'nullable|string|min:3|max:16|unique:academies,referalCode',
            'cereficate' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',
            'logo' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',
            'picture' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',
        ]);


        $referalCode = $request->input('referalCode');
        if (!$request->input('referalCode')) {
            $academy = academy::latest()->first();
            if (isset($academy->id)) {
                $id = $academy->id;
            } else {
                $id = random_int(1, 999);
            }
            $referalCode = Str::random(3) . $id . Str::random(3);
        }


        academy::create([
            'area_id' => $request->input('area_id'),
            'firstNameOwner' => $request->input('firstNameOwner'),
            'lastNameOwner' => $request->input('lastNameOwner'),
            'nationalCodeOwner' => $request->input('nationalCodeOwner'),
            'birthdayOwner' => $request->input('birthdayOwner'),
            'birthplaceOwner' => $request->input('birthplaceOwner'),
            'mobileOwner' => $request->input('mobileOwner'),
            'nationalCardOwner' => storeFile('nationalCardOwner', 'academies'),
            'firstNameAdmin' => $request->input('firstNameAdmin'),
            'lastNameAdmin' => $request->input('lastNameAdmin'),
            'nationalCodeAdmin' => $request->input('nationalCodeAdmin'),
            'birthdayAdmin' => $request->input('birthdayAdmin'),
            'birthplaceAdmin' => $request->input('birthplaceAdmin'),
            'mobileAdmin' => $request->input('mobileAdmin'),
            'nationalCardAdmin' => storeFile('nationalCardAdmin', 'academies'),

            'user_id' => $request->input('user_id'),
            'title' => $request->input('title'),
            'email' => $request->input('email'),
            'fax' => $request->input('fax'),
            'website' => $request->input('website'),
            'telephone' => $request->input('telephone'),
            'address' => $request->input('address'),
            'postalCode' => $request->input('postalCode'),
            'telegram' => $request->input('telegram'),
            'whatsapp' => $request->input('whatsapp'),
            'referalCode' => $referalCode,
            'cereficate' => storeFile('cereficate', 'academies'),
            'logo' => storeFile('logo', 'academies'),
            'picture' => storeFile('picture', 'academies'),
            'payPrice' => $request->has('payPrice'),
            'payStatus' => $request->has('payStatus'),
            'status' => $request->has('status'),
        ]);

        message();
        return redirect()->route('admin.academy.academy.index');
    }

    public function show(agent $agent)
    {
        $this->authorize('academy_show');

        return response()->json($agent);
    }

    public function edit(academy $academy)
    {
        $this->authorize('academy_update');

        return view('admin.Academy.agent.edit', compact('academy'));
    }

    public function update(Request $request, academy $academy)
    {
        $this->authorize('academy_update');

        $request->offsetSet('birthdayOwner',toGorgian($request->input('birthdayOwner')));
        $request->offsetSet('birthdayAdmin',toGorgian($request->input('birthdayAdmin')));

        $this->validate($request, [
            'area_id' => 'required|numeric|exists:areas,id',
            'firstNameOwner' => 'nullable|string|min:2|max:256',
            'lastNameOwner' => 'nullable|string|min:2|max:256',
            'nationalCodeOwner' => ['nullable', 'numeric', new nationalCode()],
            'birthdayOwner' => 'nullable|date',
            'birthplaceOwner' => 'nullable|string|min:5|max:256',
            'mobileOwner' => ['nullable', 'numeric', new mobileValidation()],
            'nationalCardOwner' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',
            'firstNameAdmin' => 'nullable|string|min:5|max:256',
            'lastNameAdmin' => 'nullable|string|min:5|max:256',
            'nationalCodeAdmin' => ['nullable', 'numeric', new nationalCode()],
            'birthdayAdmin' => 'nullable|date',
            'birthplaceAdmin' => 'nullable|string|min:5|max:256',
            'mobileAdmin' => ['nullable', 'numeric', new mobileValidation()],
            'nationalCardAdmin' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',

            'user_id' => 'required|numeric|exists:users,id',
            'title' => 'required|string|min:1|max:1024',
            'email' => 'nullable|string|email|min:3|max:512',
            'fax' => 'nullable|digits_between:4,32',
            'website' => 'nullable|string|min:1|max:512',
            'telephone' => 'required|digits_between:10,16',
            'address' => 'required|string|min:1|max:2048',
            'postalCode' => 'required|digits_between:9,16',
            'telegram' => 'nullable|string|min:1|max:256',
            'whatsapp' => 'nullable|digits_between:10,16',
            'referalCode' => [
                'required',
                'string',
                'min:3',
                'max:255',
                Rule::unique('academies')->ignore($academy->referalCode, 'referalCode'),
            ],
            'cereficate' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',
            'logo' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',
            'picture' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',
        ]);


        $academy->update([
            'area_id' => $request->input('area_id'),
            'firstNameOwner' => $request->input('firstNameOwner'),
            'lastNameOwner' => $request->input('lastNameOwner'),
            'nationalCodeOwner' => $request->input('nationalCodeOwner'),
            'birthdayOwner' => $request->input('birthdayOwner'),
            'birthplaceOwner' => $request->input('birthplaceOwner'),
            'mobileOwner' => $request->input('mobileOwner'),
            'nationalCardOwner' => updateFile('nationalCardOwner', 'academies',$academy->nationalCardOwner),
            'firstNameAdmin' => $request->input('firstNameAdmin'),
            'lastNameAdmin' => $request->input('lastNameAdmin'),
            'nationalCodeAdmin' => $request->input('nationalCodeAdmin'),
            'birthdayAdmin' => $request->input('birthdayAdmin'),
            'birthplaceAdmin' => $request->input('birthplaceAdmin'),
            'mobileAdmin' => $request->input('mobileAdmin'),
            'nationalCardAdmin' => updateFile('nationalCardAdmin', 'academies',$academy->nationalCardAdmin),

            'user_id' => $request->input('user_id'),
            'title' => $request->input('title'),
            'email' => $request->input('email'),
            'fax' => $request->input('fax'),
            'website' => $request->input('website'),
            'telephone' => $request->input('telephone'),
            'address' => $request->input('address'),
            'postalCode' => $request->input('postalCode'),
            'telegram' => $request->input('telegram'),
            'whatsapp' => $request->input('whatsapp'),
            'referalCode' => $request->input('referalCode'),
            'cereficate' => updateFile('cereficate', 'academies',$academy->cereficate),
            'logo' => updateFile('logo', 'academies',$academy->logo),
            'picture' => updateFile('picture', 'academies',$academy->picture),
            'payPrice' => $request->has('payPrice'),
            'payStatus' => $request->has('payStatus'),
            'status' => $request->has('status'),
        ]);


        message('ویرایش شد.');
        return redirect()->route('admin.academy.academy.edit', $academy->id);
    }

    public function destroy(academy $academy)
    {
        $this->authorize('academy_destroy');

        if ($academy->cereficate) {
            Storage::delete($academy->cereficate);
        }
        if ($academy->logo) {
            Storage::delete($academy->logo);
        }
        if ($academy->picture) {
            Storage::delete($academy->picture);
        }
        if ($academy->nationalCardOwner) {
            Storage::delete($academy->nationalCardOwner);
        }
        if ($academy->nationalCardAdmin) {
            Storage::delete($academy->nationalCardAdmin);
        }

        return response()->json($academy->delete());
    }
}
