<?php

namespace App\Http\Controllers\Admin\Academy;

use App\Http\Controllers\Controller;
use App\Models\Academy\agent;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class AgentController extends Controller
{
    public function index()
    {
        $this->authorize('agent_index');

        if (request()->has('page')) {
            $list = agent::search()->withCount('academy')->latest()->paginate();

            return response()->json($list);
        }

        return view('admin.Academy.agent.index');
    }

    public function create()
    {
        $this->authorize('agent_store');

        return view('admin.Academy.agent.create');
    }

    public function store(Request $request)
    {
        $this->authorize('agent_store');

        $this->validate($request, [
            'user_id' => 'required|numeric|exists:users,id',
            'title' => 'required|string|min:1|max:1024',
            'email' => 'nullable|string|email|min:3|max:512',
            'fax' => 'nullable|digits_between:4,32',
            'website' => 'nullable|string|min:1|max:512',
            'telephone' => 'required|digits_between:10,16',
            'address' => 'required|string|min:1|max:2048',
            'postalCode' => 'required|digits_between:9,16',
            'telegram' => 'nullable|string|min:1|max:256',
            'whatsapp' => 'nullable|digits_between:10,16',
            'referalCode' => 'nullable|string|min:3|max:16|unique:agents,referalCode',
            'cereficate' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',
            'logo' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',
            'picture' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',
        ]);

        $referalCode = $request->input('referalCode');
        if (!$request->input('referalCode')) {
            $agent = agent::latest()->first();
            if (isset($agent->id)) {
                $id = $agent->id;
            } else {
                $id = random_int(1, 999);
            }
            $referalCode = Str::random(3) . $id . Str::random(3);
        }


        agent::create([
            'user_id' => $request->input('user_id'),
            'title' => $request->input('title'),
            'email' => $request->input('email'),
            'fax' => $request->input('fax'),
            'website' => $request->input('website'),
            'telephone' => $request->input('telephone'),
            'address' => $request->input('address'),
            'postalCode' => $request->input('postalCode'),
            'telegram' => $request->input('telegram'),
            'whatsapp' => $request->input('whatsapp'),
            'referalCode' => $referalCode,
            'cereficate' => storeFile('cereficate', 'agents'),
            'logo' => storeFile('logo', 'agents'),
            'picture' => storeFile('picture', 'agents'),
            'payPrice' => $request->has('payPrice'),
            'payStatus' => $request->has('payStatus'),
            'status' => $request->has('status'),
        ]);

        message();
        return redirect()->route('admin.academy.agent.index');
    }

    public function show(agent $agent)
    {
        $this->authorize('agent_show');

        return response()->json($agent);
    }

    public function edit(agent $agent)
    {
        $this->authorize('agent_update');

        return view('admin.Academy.agent.edit', compact('agent'));
    }

    public function update(Request $request, agent $agent)
    {
        $this->authorize('agent_update');

        $this->validate($request, [
            'user_id' => 'required|numeric|exists:users,id',
            'title' => 'required|string|min:1|max:1024',
            'email' => 'nullable|string|email|min:3|max:512',
            'fax' => 'nullable|digits_between:4,32',
            'website' => 'nullable|string|min:1|max:512',
            'telephone' => 'required|digits_between:10,16',
            'address' => 'required|string|min:1|max:2048',
            'postalCode' => 'required|digits_between:9,16',
            'telegram' => 'nullable|string|min:1|max:256',
            'whatsapp' => 'nullable|digits_between:10,16',
            'referalCode' => [
                'required',
                'string',
                'min:3',
                'max:255',
                Rule::unique('agents')->ignore($agent->referalCode, 'referalCode'),
            ],
            'cereficate' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',
            'logo' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',
            'picture' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',
        ]);

        $agent->update([
            'user_id' => $request->input('user_id'),
            'title' => $request->input('title'),
            'email' => $request->input('email'),
            'fax' => $request->input('fax'),
            'website' => $request->input('website'),
            'telephone' => $request->input('telephone'),
            'address' => $request->input('address'),
            'postalCode' => $request->input('postalCode'),
            'telegram' => $request->input('telegram'),
            'whatsapp' => $request->input('whatsapp'),
            'referalCode' => $request->input('referalCode'),
            'cereficate' => updateFile('cereficate', 'agents',$agent->cereficate),
            'logo' => updateFile('logo', 'agents',$agent->logo),
            'picture' => updateFile('picture', 'agents',$agent->picture),
            'payPrice' => $request->has('payPrice'),
            'payStatus' => $request->has('payStatus'),
            'status' => $request->has('status'),
        ]);


        message('ویرایش شد.');
        return redirect()->route('admin.academy.agent.edit', $agent->id);
    }

    public function destroy(agent $agent)
    {
        $this->authorize('agent_destroy');

        if ($agent->cereficate) {
            Storage::delete($agent->cereficate);
        }
        if ($agent->logo) {
            Storage::delete($agent->logo);
        }
        if ($agent->picture) {
            Storage::delete($agent->picture);
        }


        return response()->json($agent->delete());
    }
}
