<?php

namespace App\Http\Controllers\Admin\Organization;

use App\Http\Controllers\Controller;
use App\Models\Organization\organization;
use Illuminate\Http\Request;

class OrganizationController extends Controller
{
    public function index()
    {
        $this->authorize('organization_index');

        if (request()->has('page')) {
            if (auth()->user()->id == 1) {
                $list = organization::search()->withCount('admins','userReferal')->latest('id')->paginate();
            } else {
                $list = organization::search()->whereNotIn('id',[1,2,3,4,5,6])->withCount('admins','userReferal')->latest()->paginate();
            }

            return response()->json($list);
        }

        return view('admin.Organization.organization.index');
    }

    public function create()
    {
        $this->authorize('organization_store');

        return view('admin.Organization.organization.create');
    }

    public function store(Request $request)
    {
        $this->authorize('organization_store');

        $this->validate($request, [
            'title' => 'required|string|max:255',
            'body' => 'nullable|string',
            'logo' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',
            'picture' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',
        ]);

        $logo = null;
        if ($request->hasFile('logo')) {
            $logo = $request->file('logo')->store('uploads/organizations');
        }

        $picture = null;
        if ($request->hasFile('picture')) {
            $picture = $request->file('picture')->store('uploads/organizations');
        }


        if (!$request->input('parent_id')) {
            $request->offsetSet('parent_id',6);
        }

        $organization = organization::create([
            'title' => $request->input('title'),
            'parent_id' => $request->input('parent_id'),
            'body' => $request->input('body'),
            'logo' => $logo,
            'picture' => $picture,
            'status' => $request->has('status')
        ]);

        $organization->wallet()->create([
            'withdraw' => 0,
            'deposit' => 0,
            'amount' => 0,
            'status' => 1
        ]);

        $randomInt = random_int(9999, 99999);
        $organization->update([
            'code' => $randomInt . $organization->id
        ]);

        $organization->admins()->sync(json_decode($request->input('admins_id')));


        message();
        return redirect()->route('admin.organization.organization.index');
    }

    public function show(state $state)
    {
        $this->authorize('organization_show');

        return response()->json($state);
    }

    public function edit(organization $organization)
    {
        $this->authorize('organization_update');

        $parent = null;
        if ($organization->parent_id != 1 && $organization->parent_id != 2 && $organization->parent_id != 3 && $organization->parent_id != 4 && $organization->parent_id != 5 && $organization->parent_id != 6) {
            $parent = $organization->parent;
        }

        return view('admin.Organization.organization.edit',compact('organization','parent'));
    }

    public function update(Request $request, organization $organization)
    {
        $this->authorize('organization_update');

        $this->validate($request, [
            'title' => 'required|string|max:255',
            'body' => 'nullable|string',
            'logo' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',
            'picture' => 'nullable|image|mimes:jpeg,bmp,png|max:4096',
        ]);

        if ($request->hasFile('logo')) {
            $organization->logo = $request->file('logo')->store('uploads/organizations');
        }

        if ($request->hasFile('picture')) {
            $organization->picture = $request->file('picture')->store('uploads/organizations');
        }

        if (!$request->input('parent_id') || $request->input('parent_id') == $organization->id) {
            $request->offsetSet('parent_id', 6);
        }

        $organization->update([
            'title' => $request->input('title'),
            'parent_id' => $request->input('parent_id'),
            'body' => $request->input('body'),
            'logo' => $organization->logo,
            'picture' => $organization->picture,
            'status' => $request->has('status')
        ]);
        $organization->admins()->sync(json_decode($request->input('admins_id')));


        message('ویرایش شد.');
        return redirect()->route('admin.organization.organization.edit',$organization->id);
    }

    public function destroy(organization $organization)
    {
        $this->authorize('organization_destroy');


        if ($organization->id != 1 && $organization->id != 2 && $organization->id != 3 && $organization->id != 4 && $organization->id != 5 && $organization->id != 6) {
            if ($organization->userReferal()->count() == 0) {
                return response()->json($organization->delete());
            }
        }

    }


    // Relations

    public function refUsers(organization $organization)
    {
        $this->authorize('organization_store');

        if (request()->has('page')) {
            $list = $organization->userReferal()->search()->paginate();

            return response()->json($list);
        }

        return view('admin.Organization.organization.refUsers');
    }

}
